/* 
 * 
 * Home Health Grouper PDGM (Patient-Driven Grouping Model)
 * 
 * Developed for the Center for Medicare and Medicaid Services (CMS)
 * by 3M Health Information Systems.
 * 
 * All code is provided as is.
 * 
 */
package com.mmm.cms.hh.pdgm.grouper;

import com.mmm.cms.hh.pdgm.data.files.TableNames_EN;
import com.mmm.cms.hh.pdgm.reference.objects.ClinicalGroupObject;
import com.mmm.cms.hh.pdgm.reference.objects.ComorbidityGroupObject;
import com.mmm.cms.hh.pdgm.reference.objects.ComorbidityInteractionObject;
import com.mmm.cms.hh.pdgm.reference.objects.DiagnosisObject;
import com.mmm.cms.hh.pdgm.reference.objects.DiagnosisSubchapterObject;
import com.mmm.cms.hh.pdgm.reference.objects.HippsValueObject;
import com.mmm.cms.hh.pdgm.reference.objects.OasisRespondObject;
import com.mmm.cms.hh.pdgm.reference.objects.ReturnCodeObject;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author US340646
 */
public class DataManager implements DataManagerIF {

    private static final Logger LOG = Logger.getLogger(DataManager.class.getName());

    /**
     *
     * @param rtBasePath
     * @param loadDescriptions
     */
    public DataManager(String rtBasePath, boolean loadDescriptions) {
        // load all rots
        loadData(rtBasePath, loadDescriptions);
    }

    /**
     *
     * @param rtBasePath
     */
    public DataManager(String rtBasePath) {
        // load all rots
        loadData(rtBasePath, false);
    }

    // load all RT tables here
    private void loadData(String rtBasePath, boolean loadDescriptions) {

        dataMap = new TreeMap();

        // Diagnoses
        String inpFile = rtBasePath + "/" + TableNames_EN.Diagnoses.getFileName();
        DataContainerIF<DiagnosisObject> diagnoses
                = new DataContainer(inpFile, 0, DiagnosisObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.Diagnoses, diagnoses);

        // Clinical Group RT
        inpFile = rtBasePath + "/" + TableNames_EN.Clinical_Groups.getFileName();
        DataContainerIF<ClinicalGroupObject> clinicalGroups
                = new DataContainer(inpFile, 0, ClinicalGroupObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.Clinical_Groups, clinicalGroups);

        // Comorbidity Group RT
        inpFile = rtBasePath + "/" + TableNames_EN.Comorbidity_Groups.getFileName();
        DataContainerIF<ComorbidityGroupObject> comorbidityGroups
                = new DataContainer(inpFile, 0, ComorbidityGroupObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.Comorbidity_Groups, comorbidityGroups);

        // Comorbidity Interactions RT
        inpFile = rtBasePath + "/" + TableNames_EN.Comorbidity_Interactions.getFileName();
        DataContainerIF<ComorbidityInteractionObject> comorbidityInteractions
                = new DataContainer(inpFile, 0, ComorbidityInteractionObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.Comorbidity_Interactions, comorbidityInteractions);

        // Diagnosis Subchapter RT
        inpFile = rtBasePath + "/" + TableNames_EN.Diagnosis_Subchapters.getFileName();
        DataContainerIF<DiagnosisSubchapterObject> diagnosisSubchapters
                = new DataContainer(inpFile, 0, DiagnosisSubchapterObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.Diagnosis_Subchapters, diagnosisSubchapters);

        // HIPPS Structure RT
        inpFile = rtBasePath + "/" + TableNames_EN.HIPPS_Structure.getFileName();
        DataContainerIF<HippsValueObject> hippsValues
                = new DataContainer(inpFile, HippsValueObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.HIPPS_Structure, hippsValues);

        // OASIS Responses RT
        inpFile = rtBasePath + "/" + TableNames_EN.OASIS_Responses.getFileName();
        DataContainerIF<OasisRespondObject> oasisResponces
                = new DataContainer(inpFile, OasisRespondObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.OASIS_Responses, oasisResponces);

        // Return Codes RT
        inpFile = rtBasePath + "/" + TableNames_EN.Return_Codes.getFileName();
        DataContainerIF<ReturnCodeObject> returnCodes
                = new DataContainer(inpFile, 0, ReturnCodeObject.class, loadDescriptions);
        dataMap.put(TableNames_EN.Return_Codes, returnCodes);

        LOG.log(Level.INFO, "Grouper Data Manager instantiated: RT path: {0}, Descriptions loaded: {1}", new Object[]{rtBasePath, loadDescriptions});
        
    }

    private Map<TableNames_EN, DataContainerIF> dataMap;

    /**
     * @return the dataMap
     */
    @Override
    public Map<TableNames_EN, DataContainerIF> getDataMap() {
        return dataMap;
    }

    /**
     * @param dataMap the dataMap to set
     */
    @Override
    public void setDataMap(Map<TableNames_EN, DataContainerIF> dataMap) {
        this.dataMap = dataMap;
    }
}
