/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.logic.processor;

import gov.cms.hh.claim.ClaimIF;
import gov.cms.hh.data.meta.enumer.EditId_EN;
import gov.cms.hh.data.meta.enumer.EditType_EN;
import gov.cms.hh.diagnosis.ClinicalGroupIF;
import gov.cms.hh.diagnosis.DiagnosisIF;
import gov.cms.hh.grouper.DataManagerIF;
import gov.cms.hh.logic.validation.Edit;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author CMS
 */
public class HippsPosition2 extends HippsPositionAbstract implements HippsPositionIF {

    private static final Logger LOG = Logger.getLogger(HippsPosition2.class.getName());

    /*
    * 2. HIPPS 2nd position
    * 2.1 Validate PDX for scorability
    * 2.2 Validate 1st SDX for scorability
    * 2.3 Calculate 2nd value
     */

    /**
     *
     * @param claim
     * @param dataManager
     */

    public HippsPosition2(final ClaimIF claim, final DataManagerIF dataManager) {
        super(claim, dataManager);
    }

    /**
     *
     * @return
     */
    @Override
    public boolean validate() {

        // Preliminary validate claim DXs
        validateDiagnoses();

        // 2.1 Validate other PDX and SDX with primary awarding flag properties for scorability        
        DiagnosisIF pdx = getClaim().getPrimaryDiagnosis();
        // PDX
        if (checkAsPdx("PDX", pdx)) {
            // SDXs
            // a. ONLY 1st SDX should be used for primary awarding
            // b. if 1st SDX is valid (including PDX criteria) then 1st SDX should move to PDX position
            List<DiagnosisIF> sdxList = getClaim().getSecondaryDiagnoses().getCollection();
            if (!sdxList.isEmpty()) {
                DiagnosisIF sdx = sdxList.get(0);
                if (sdx.isPrimaryAwarding()) {
                    if (checkAsPdx("SDX1 as PDX", sdx)) {
                        // replace PDX
                        pdx.setValue(sdx.getValue());
                        pdx.setPrimaryAwarding(sdx.isPrimaryAwarding());
                        pdx.setManifestation(sdx.isManifestation());
                        pdx.setId(sdx.getId());
                        pdx.setECOI(sdx.isECOI());
                        pdx.setCodeFirst(sdx.isCodeFirst());
                        pdx.setUnacceptablePdx(sdx.isUnacceptablePdx());
                        pdx.setDescription(sdx.getDescription());
                        pdx.setDiagnosisSubchapter(sdx.getDiagnosisSubchapter());
                        pdx.setComorbidityGroup(sdx.getComorbidityGroup());
                        pdx.setClinicalGroup(sdx.getClinicalGroup());
                        // add edit
                        pdx.addEdit(
                                new Edit(EditId_EN.VALUE_SUBSTITUTED, pdx + " "
                                        + EditId_EN.VALUE_SUBSTITUTED.getDescription(), EditType_EN.SPECIFICATION, Level.INFO)
                        );
                        sdx.addEdit(
                                new Edit(EditId_EN.VALUE_PROMOTED, sdx + " "
                                        + EditId_EN.VALUE_PROMOTED.getDescription(), EditType_EN.SPECIFICATION, Level.INFO)
                        );
                        sdx.addEdit(
                                new Edit(EditId_EN.VALUE_IGNORE, sdx + " "
                                        + EditId_EN.VALUE_IGNORE.getDescription(), EditType_EN.SPECIFICATION, Level.INFO)
                        );
                    }
                }
            }
            return true;
        } else {
            return false;
        }
    }

    /**
     *
     * @return
     */
    @Override
    public boolean calculate() {

        final int HIPPS_POSITION = 2;

        String text = null;

        DiagnosisIF pdx = getClaim().getPrimaryDiagnosis();

        if (!pdx.getEditCollection().hasCriticalEdits()) {

            // Set HIPPS 2
            // Add logic info edits for Detail Report
            // PDX info
            text = "PDX = " + pdx.getValue() + " (" + pdx.getDescription() + ")"
                    + (pdx.getEditCollection().hasEditId(EditId_EN.VALUE_SUBSTITUTED)? " (SDX promoted to PDX)" : "");
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            // Clinical Group info
            text = "PDX Clinical Group = " + pdx.getClinicalGroup();
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            // PDX Subchapter info
            text = "PDX Subchapter = " + pdx.getDiagnosisSubchapter().getDxFrom()
                    + "-" + pdx.getDiagnosisSubchapter().getDxTo()
                    + " (" + pdx.getDiagnosisSubchapter().getDescription() + ")";

            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            text = "HIPPS Position 2 = " + pdx.getClinicalGroup().getValue() + " (" + pdx.getClinicalGroup().getDescription() + ")";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            return setHippsCode(HIPPS_POSITION, "" + pdx.getClinicalGroup().getValue());
        } else {

            // Add logic info edits for Detail Report
            text = "PDX has critical issue(s). Please see validation messages";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            text = "HIPPS Position 2 = 0 (Not calculated)";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            return false;
        }

    }

    /**
     *
     * @param label
     * @param pdx
     * @return
     */
    protected boolean checkAsPdx(String label, DiagnosisIF pdx) {

        final int GRC_INVALID_PDX = 3;
        final int GRC_UNGROUPABLE_PDX = 5;

        boolean retFlag = true;

        if (!pdx.getEditCollection().hasEditId(EditId_EN.VALUE_INVALID)) {
            // check Clinical Group
            ClinicalGroupIF cg = pdx.getClinicalGroup();
            if (cg == null || cg.getValue().isEmpty() || cg.getValue().equalsIgnoreCase("na")) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " has no Clinical Group assigned, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                addReturnCode(GRC_UNGROUPABLE_PDX);
                retFlag = false;
            }
            // check if manifestation
            if (pdx.isManifestation()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is manifestation DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_UNGROUPABLE_PDX);
                retFlag = false;
            }
            // check if ECOI
            if (pdx.isECOI()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is ECOI DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_UNGROUPABLE_PDX);
                retFlag = false;
            }
            // check if Unacceptable PDX
            if (pdx.isUnacceptablePdx()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is unacceptable as PDX DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_UNGROUPABLE_PDX);
                retFlag = false;
            }
        } else {
            addReturnCode(GRC_INVALID_PDX);
            retFlag = false;
        }
        return retFlag;
    }

}
