/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.v04023;

import gov.cms.hh.claim.ClaimIF;
import gov.cms.hh.data.exchange.ClaimContainerIF;
import gov.cms.hh.data.files.TableNames_EN;
import gov.cms.hh.data.meta.enumer.EditId_EN;
import gov.cms.hh.data.meta.enumer.EditType_EN;
import gov.cms.hh.diagnosis.DiagnosisIF;
import gov.cms.hh.grouper.DataManagerIF;
import gov.cms.hh.grouper.GrouperIF;
import gov.cms.hh.logic.processor.HippsProcessorIF;
import gov.cms.hh.logic.specification.IsCodeValid;
import gov.cms.hh.logic.specification.IsSpanClaim;
import gov.cms.hh.logic.validation.Edit;
import gov.cms.hh.reference.objects.DiagnosisObject;
import gov.cms.hh.v03322.*;
import gov.cms.hh.reference.objects.GrouperVersionObject;

import java.text.ParseException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author CMS
 */
public class Grouper_v04023 extends Grouper_v03322 {

    /**
     *
     * @param id
     * @param grouperVersionObject
     * @param loadDescription
     * @throws ParseException
     */
    public Grouper_v04023(Integer id, GrouperVersionObject grouperVersionObject, Boolean loadDescription) throws ParseException {
        super(id, grouperVersionObject, loadDescription);

        /**
         * Set for using link to next version for spanning claims Requirements
         * for April 2023:
         */
        setLinked();
    }

    /**
     *
     * @param claim
     * @param dataManager
     */
    @Override
    protected void processClaim(final ClaimIF claim, final DataManagerIF dataManager) {
        HippsProcessorIF hp;
        hp = new HippsProcessor_v04023();
        hp.process(claim, dataManager);
    }
    
    /**
     * Claim caster method for potential span claims HHGS requirements for April
     * 2023
     *
     * @param claimInput
     * @return
     */
    @Override
    protected ClaimIF castClaim(final ClaimContainerIF claimInput) {
        ClaimIF claim = super.castClaim(claimInput);
        // Check for span claim and potential PDX substitution
        // Specs for v04.1.23 in relation to v04.0.23
        // Is valid PDX?
        DiagnosisIF pdx = claim.getPrimaryDiagnosis();
        boolean isPdxValid = (new IsCodeValid(pdx.getValue(), TableNames_EN.Diagnoses)).isSatisfiedBy(this);
        if (isPdxValid) {
            return claim;
        }
        // is linked version?
        if (!isLinked()) {
            return claim;
        }
        // is span claim?
        boolean isSpanClaim = (new IsSpanClaim(claim.getDateFrom().getValue())).isSatisfiedBy(this);
        if(!isSpanClaim) {
            return claim;
        }
        // OK, check next version
        boolean isPdxNextValid = (new IsCodeValid(pdx.getValue(), TableNames_EN.Diagnoses)).isSatisfiedBy(getNextVersion());
        if (isPdxNextValid) {
            // try to get the predecessor DX
            String predecessorDx = getPredecessorDx(pdx.getValue());
            // substitute PDX with predecessor DX
            pdx.setValue(predecessorDx);
            pdx.addEdit(new Edit(EditId_EN.VALUE_SUBSTITUTED_FOR_SPAN, pdx + ", "
                    + EditId_EN.VALUE_SUBSTITUTED_FOR_SPAN.getDescription(),
                                EditType_EN.SPECIFICATION, Level.INFO));
        }
        // Return claim with updated PDX (if found)
        return claim;
    }

    /**
     *
     * @param dx
     * @return
     */
    protected String getPredecessorDx(String dx) {
        GrouperIF nextGrouper = getNextVersion();
        DiagnosisObject dxObj = (DiagnosisObject)nextGrouper.getAvailableTableContent(
                                TableNames_EN.Diagnoses).getDataMap().get(dx);
        String predecessor = null;
        try {
            predecessor = dxObj.getPredecessorDx();
        } catch (Exception e) {
            LOG.info("*** SEVERE: Grouper Internal Error: Predecessor DX not found ***");
        }
        return predecessor;
    }
    
    private static final Logger LOG = Logger.getLogger(Grouper_v04023.class.getName());

}
