/* 
 * 
 * Home Health Grouper PDGM (Patient-Driven Grouping Model)
 * 
 * Developed for the Center for Medicare and Medicaid Services (CMS)
 * by 3M Health Information Systems.
 * 
 * All code is provided as is.
 * 
 */
package com.mmm.cms.hh.pdgm.logic.processor;

import com.mmm.cms.hh.pdgm.claim.ClaimIF;
import com.mmm.cms.hh.pdgm.data.meta.enumer.EditId_EN;
import com.mmm.cms.hh.pdgm.data.meta.enumer.EditType_EN;
import com.mmm.cms.hh.pdgm.diagnosis.ClinicalGroup;
import com.mmm.cms.hh.pdgm.diagnosis.DiagnosisIF;
import com.mmm.cms.hh.pdgm.grouper.DataManagerIF;
import com.mmm.cms.hh.pdgm.logic.validation.Edit;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author us340646
 */
public class HippsPosition2 extends HippsPositionAbstract implements HippsPositionIF {

    private static final Logger LOG = Logger.getLogger(HippsPosition2.class.getName());

    /*
    * 2. HIPPS 2nd position
    * 2.1 Validate PDX for scorability
    * 2.2 Validate 1st SDX for scorability
    * 2.3 Calculate 2nd value
     */

    /**
     *
     * @param claim
     * @param dataManager
     */

    public HippsPosition2(ClaimIF claim, DataManagerIF dataManager) {
        super(claim, dataManager);
    }

    /**
     *
     * @param visitor
     */
    @Override
    public void process(HippsPositionSelectorIF visitor) {
        visitor.select(this);
    }

    /**
     *
     * @return
     */
    @Override
    public boolean validate() {

        // Preliminary validate claim DXs
        validateDiagnoses();

        // 2.1 Validate other PDX and SDX with primary awarding flag properties for scorability        
        DiagnosisIF pdx = getClaim().getPrimaryDiagnosis();
        // PDX
        if (checkAsPdx("PDX", pdx)) {
            // SDXs
            // a. ONLY 1st SDX should be used for primary awarding
            // b. if 1st SDX is valid (including PDX criteria) then 1st SDX should move to PDX position
            List<DiagnosisIF> sdxList = getClaim().getSecondaryDiagnoses();
            if (!sdxList.isEmpty()) {
                DiagnosisIF diag = sdxList.get(0);
                if (diag.isPrimaryAwarding()) {
                    if (checkAsPdx("SDX1 as PDX", diag)) {
                        getClaim().setPrimaryDiagnosis(diag);
                        sdxList.remove(0);
                        // add edit
                        getClaim().getPrimaryDiagnosis().addEdit(
                                new Edit(EditId_EN.VALUE_SUBSTITUTED, pdx + " "
                                        + EditId_EN.VALUE_SUBSTITUTED.getDescription(), EditType_EN.SPECIFICATION, Level.INFO)
                        );
                    }
                }
            }
            return true;
        } else {
            return false;
        }
    }

    /**
     *
     * @return
     */
    @Override
    public boolean calculate() {

        final int HIPPS_POSITION = 2;

        String text = null;

        DiagnosisIF pdx = getClaim().getPrimaryDiagnosis();

        if (!pdx.getEditCollection().hasCriticalEdits()) {

            // Set HIPPS 2
            // Add logic info edits for Detail Report
            // PDX info
            text = "PDX = " + pdx.getValue() + " (" + pdx.getDescription() + ")"
                    + (pdx.getEditCollection().hasEditId(EditId_EN.VALUE_SUBSTITUTED)? " (Primary awarded SDX1)" : "");
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            // Clinical Group info
            text = "PDX Clincal Group = " + pdx.getClinicalGroup();
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            // PDX Subchapter info
            text = "PDX Subchapter = " + pdx.getDiagnosisSubchapter().getDxFrom()
                    + "-" + pdx.getDiagnosisSubchapter().getDxTo()
                    + " (" + pdx.getDiagnosisSubchapter().getDescription() + ")";

            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            text = "HIPPS Position 2 = " + pdx.getClinicalGroup().getValue() + " (" + pdx.getClinicalGroup().getDescription() + ")";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            return setHippsCode(HIPPS_POSITION, "" + pdx.getClinicalGroup().getValue());
        } else {

            // Add logic info edits for Detail Report
            text = "PDX has critical issue(s). Please see validation messages";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            text = "HIPPS Position 2 = 0 (Not calculated)";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS2, text);

            return false;
        }

    }

    private boolean checkAsPdx(String label, DiagnosisIF pdx) {

        final int GRC_INVALID_PDX = 3;
        final int GRC_UNGROUPABLE_PDX = 5;

        boolean retFlag = true;

        if (!pdx.getEditCollection().hasEditId(EditId_EN.VALUE_INVALID)) {
            // check Clinical Group
            ClinicalGroup cg = pdx.getClinicalGroup();
            if (cg == null || cg.getValue().isEmpty() || cg.getValue().equalsIgnoreCase("n/a")) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " has no Clinical Group assigned, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                addReturnCode(GRC_UNGROUPABLE_PDX);
                retFlag = false;
            }
            // check if manifestation
            if (pdx.isManifestation()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is manifestation DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_UNGROUPABLE_PDX);
                // add validation flag
//                getClaim().getValidityFlags().add(new ValidityFlag("03", label + " " + pdx + " is manifestation DX"));
                retFlag = false;
            }
            // check if ECOI
            if (pdx.isECOI()) {
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_UNGROUPABLE,
                                label + " " + pdx + " is ECOI DX, "
                                + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
                );
                // add return code
                addReturnCode(GRC_UNGROUPABLE_PDX);
                // add validation flag
//                getClaim().getValidityFlags().add(new ValidityFlag("03", label + " " + pdx + " is ECOI DX"));
                retFlag = false;
            }
        } else {
            addReturnCode(GRC_INVALID_PDX);
            retFlag = false;
        }
        return retFlag;
    }

}
