/* 
 * 
 * Home Health Grouper PDGM (Patient-Driven Grouping Model)
 * 
 * Developed for the Center for Medicare and Medicaid Services (CMS)
 * by 3M Health Information Systems.
 * 
 * All code is provided as is.
 * 
 */
package com.mmm.cms.hh.pdgm.logic.processor;

import com.mmm.cms.hh.pdgm.claim.ClaimIF;
import com.mmm.cms.hh.pdgm.claim.OasisResponseIF;
import com.mmm.cms.hh.pdgm.data.files.TableNames_EN;
import com.mmm.cms.hh.pdgm.data.meta.enumer.EditId_EN;
import com.mmm.cms.hh.pdgm.data.meta.enumer.FunctionalLevel_EN;
import com.mmm.cms.hh.pdgm.diagnosis.DiagnosisIF;
import com.mmm.cms.hh.pdgm.grouper.DataManagerIF;
import com.mmm.cms.hh.pdgm.reference.objects.OasisRespondObject;
import java.util.List;

/**
 *
 * @author us340646
 */
public class HippsPosition3 extends HippsPositionAbstract implements HippsPositionIF {

    /*
    * 3. HIPPS 3rd position
    * 3.1 Validate OASIS responses
    * 3.2 Calculate 3rd value
     */

    /**
     *
     * @param claim
     * @param dataManager
     */

    public HippsPosition3(ClaimIF claim, DataManagerIF dataManager) {
        super(claim, dataManager);
    }

    /**
     *
     * @param visitor
     */
    @Override
    public void process(HippsPositionSelectorIF visitor) {
        visitor.select(this);
    }

    /**
     *
     * @return
     */
    @Override
    public boolean validate() {
        // Validate all functional fields
        return validateFunctionalFields();
    }

    /**
     *
     * @return
     */
    @Override
    public boolean calculate() {

        final int HIPPS_POSITION = 3;

        DiagnosisIF pdx = getClaim().getPrimaryDiagnosis();
        if (!(getClaim().getOasisFields().getEditCollection().hasCriticalEdits()
                || pdx.getEditCollection().hasCriticalEdits())) {

            // Calculate total functional score
            int score = getRiskHospitalisationScore() + getOtherFunctionalScore();

            // Total functional impairment points
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS3, "Total Functional Impairment points " + score);

            // Calculate HIPPS 3 based on total functional score and clinical group
            String value = getHippsValue(score);
            
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS3, "HIPPS Position 3 = " + value 
                    + " (" + FunctionalLevel_EN.valueOf(value).getName() + ", see PDX clinical group low/high value range)");

            // Set HIPPS 3
            return setHippsCode(HIPPS_POSITION, value);

        } else {
            
            // Add logic info edits for Detail Report
            String text = "Functional Impairment Fields or PDX have critical issue(s). Please see validation messages";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS3, text);

            text = "HIPPS Position 3 = 0 (Not calculated)";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS3, text);

            return false;
        }
    }

    private String getHippsValue(int score) {
        String retVal = null;
        int lowPoints = getClaim().getPrimaryDiagnosis().getClinicalGroup().getLowPoints();
        int highPoints = getClaim().getPrimaryDiagnosis().getClinicalGroup().getHighPoints();

        if (score <= lowPoints) {
            retVal = FunctionalLevel_EN.A.name();
        } else if (score >= highPoints) {
            retVal = FunctionalLevel_EN.C.name();
        } else {
            retVal = FunctionalLevel_EN.B.name();
        }
        return retVal;
    }

    private int getOtherFunctionalScore() {

        final String PREFIX_NOT = "M1033";

        // Sum of M18xx (other) responses
        int sum = 0;
        List<OasisResponseIF> oResList = getClaim().getOasisFields();
        for (OasisResponseIF oResp : oResList) {
            if (!oResp.getType().name().startsWith(PREFIX_NOT)) {
                sum += oResp.getScore();
                addHippsLogicMessage(EditId_EN.VALUE_HIPPS3, "OASIS " + oResp.getType().name() + " points = " + oResp.getScore());
            }
        }

        return sum;

    }

    private int getRiskHospitalisationScore() {

        int retScore = 0;

        final String PREFIX = "M1033";

        // Get RT data
        List<OasisRespondObject> refList = getDataManager().getDataMap().get(TableNames_EN.OASIS_Responses).getData();
        OasisRespondObject ref = null;
        for (OasisRespondObject oasRespObj : refList) {
            if (oasRespObj.getFieldId().startsWith(PREFIX)) {
                ref = oasRespObj;
            }
        }

        // Sum of M1033 responses
        int sum = 0;
        List<OasisResponseIF> oResList = getClaim().getOasisFields();
        for (OasisResponseIF oResp : oResList) {
            if (oResp.getType().name().startsWith(PREFIX)) {
                if (!oResp.getEditCollection().hasEditId(EditId_EN.VALUE_IGNORE)) {
                    int val = Integer.parseInt(oResp.getValue());
                    sum += val;
                }
            }
        }

        // Calc M1033 score
        if (ref != null) {
            if (sum >= Integer.parseInt(ref.getResponseLimit())) {
                retScore = Integer.parseInt(ref.getCategoryPoints());
            }
        }
        addHippsLogicMessage(EditId_EN.VALUE_HIPPS3, "OASIS M1033 Hospitalization Risk points total = " + retScore);
        return retScore;
    }

}
