/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.grouper;

import gov.cms.hh.data.files.TableNames_EN;
import gov.cms.hh.reference.objects.ClinicalGroupObject;
import gov.cms.hh.reference.objects.CodeFirstConditionObject;
import gov.cms.hh.reference.objects.ComorbidityGroupObject;
import gov.cms.hh.reference.objects.ComorbidityInteractionObject;
import gov.cms.hh.reference.objects.DiagnosisObject;
import gov.cms.hh.reference.objects.DiagnosisSubchapterObject;
import gov.cms.hh.reference.objects.HippsValueObject;
import gov.cms.hh.reference.objects.OasisRespondObject;
import gov.cms.hh.reference.objects.ReturnCodeObject;
import gov.cms.hh.reference.objects.ValidityFlagObject;
import java.io.InputStream;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Logger;

/**
 *
 * @author CMS
 */
public class DataManager implements DataManagerIF {

    private static final Logger LOG = Logger.getLogger(DataManager.class.getName());
    /* 2023.05.03: Fix for multithreaded issue reported by GrouperApps*/
    private final Object lock = new Object();

    /**
     *
     * @param rtBasePath
     * @param loadDescriptions
     */
    public DataManager(String rtBasePath, boolean loadDescriptions) {
        this.rtBasePath = rtBasePath;
        this.loadDescriptions = loadDescriptions;
    }

    /**
     *
     * @param rtBasePath
     */
    public DataManager(String rtBasePath) {
        // ctor chain
        this(rtBasePath, false);
    }

    // load ROT tables here
    private void loadRotData() {

        dataMap = new TreeMap();
        
        // Diagnoses
        String inpFile = getRtBasePath() + "/" + TableNames_EN.Diagnoses.getFileName();
        DataContainerIF<DiagnosisObject> diagnoses
                = new DataContainer(inpFile, 0, DiagnosisObject.class, isLoadDescriptions());
        dataMap.put(TableNames_EN.Diagnoses, diagnoses);

        // Clinical Group RT
        inpFile = getRtBasePath() + "/" + TableNames_EN.Clinical_Groups.getFileName();
        DataContainerIF<ClinicalGroupObject> clinicalGroups
                = new DataContainer(inpFile, 0, ClinicalGroupObject.class, isLoadDescriptions());
        dataMap.put(TableNames_EN.Clinical_Groups, clinicalGroups);

        // Comorbidity Group RT
        inpFile = getRtBasePath() + "/" + TableNames_EN.Comorbidity_Groups.getFileName();
        DataContainerIF<ComorbidityGroupObject> comorbidityGroups
                = new DataContainer(inpFile, 0, ComorbidityGroupObject.class, isLoadDescriptions());
        dataMap.put(TableNames_EN.Comorbidity_Groups, comorbidityGroups);

        // Comorbidity Interactions RT
        inpFile = getRtBasePath() + "/" + TableNames_EN.Comorbidity_Interactions.getFileName();
        DataContainerIF<ComorbidityInteractionObject> comorbidityInteractions
                = new DataContainer(inpFile, 0, ComorbidityInteractionObject.class, isLoadDescriptions());
        dataMap.put(TableNames_EN.Comorbidity_Interactions, comorbidityInteractions);

        // Diagnosis Subchapter RT
        inpFile = getRtBasePath() + "/" + TableNames_EN.Diagnosis_Subchapters.getFileName();
        DataContainerIF<DiagnosisSubchapterObject> diagnosisSubchapters
                = new DataContainer(inpFile, 0, DiagnosisSubchapterObject.class, isLoadDescriptions());
        dataMap.put(TableNames_EN.Diagnosis_Subchapters, diagnosisSubchapters);

        // HIPPS Structure RT
        inpFile = getRtBasePath() + "/" + TableNames_EN.HIPPS_Structure.getFileName();
        DataContainerIF<HippsValueObject> hippsValues
                = new DataContainer(inpFile, HippsValueObject.class, true); // Grouper result (HIPPS code) descriptions are not disableable
        dataMap.put(TableNames_EN.HIPPS_Structure, hippsValues);

        // OASIS Responses RT
        inpFile = getRtBasePath() + "/" + TableNames_EN.OASIS_Responses.getFileName();
        DataContainerIF<OasisRespondObject> oasisResponces
                = new DataContainer(inpFile, OasisRespondObject.class, isLoadDescriptions());
        dataMap.put(TableNames_EN.OASIS_Responses, oasisResponces);

        // Return Codes RT
        inpFile = getRtBasePath() + "/" + TableNames_EN.Return_Codes.getFileName();
        DataContainerIF<ReturnCodeObject> returnCodes
                = new DataContainer(inpFile, 0, ReturnCodeObject.class, true); // GRC descriptions are not disableable
        dataMap.put(TableNames_EN.Return_Codes, returnCodes);

        // Validity_Flags RT effective from v04.1.23
        inpFile = getRtBasePath() + "/" + TableNames_EN.Validity_Flags.getFileName();
        InputStream inStr = this.getClass().getClassLoader().getResourceAsStream(inpFile);
        if (inStr != null) {
            DataContainerIF<ValidityFlagObject> validityFlags
                    = new DataContainer(inpFile, 0, ValidityFlagObject.class, true); // Validity Flag descriptions are not disableable
            dataMap.put(TableNames_EN.Validity_Flags, validityFlags);
        }

        // Code First Conditions RT effective from v04.1.23
        inpFile = getRtBasePath() + "/" + TableNames_EN.CodeFirst_Conditions.getFileName();
        inStr = this.getClass().getClassLoader().getResourceAsStream(inpFile);
        if (inStr != null) {
            DataContainerIF<CodeFirstConditionObject> codeFirstConditions
                    = new DataContainer(inpFile, 0, CodeFirstConditionObject.class, isLoadDescriptions());
            dataMap.put(TableNames_EN.CodeFirst_Conditions, codeFirstConditions);
        }

    }

    private Map<TableNames_EN, DataContainerIF> dataMap;
    private String rtBasePath;
    private boolean loadDescriptions;

    /**
     * @return the dataMap
     */
    @Override
    public Map<TableNames_EN, DataContainerIF> getDataMap() {
        // load all rots via lazy initialization
        /* 2023.05.03: Fix for multithreaded issue reported by GrouperApps */
        synchronized (lock) {
            if (dataMap == null || dataMap.isEmpty()) {
                loadRotData();
            }
        }
        return dataMap;
    }

    /**
     * @param dataMap the dataMap to set
     */
    @Override
    public void setDataMap(Map<TableNames_EN, DataContainerIF> dataMap) {
        this.dataMap = dataMap;
    }

    /**
     * @return the loadDescriptions
     */
    @Override
    public boolean isLoadDescriptions() {
        return loadDescriptions;
    }

    /**
     * @param loadDescriptions the loadDescriptions to set
     */
    public void setLoadDescriptions(boolean loadDescriptions) {
        this.loadDescriptions = loadDescriptions;
    }

    /**
     * @return the rtBasePath
     */
    public String getRtBasePath() {
        return rtBasePath;
    }

    /**
     * @param rtBasePath the rtBasePath to set
     */
    public void setRtBasePath(String rtBasePath) {
        this.rtBasePath = rtBasePath;
    }
}
