/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.logic.processor;

import gov.cms.hh.claim.ClaimIF;
import gov.cms.hh.data.meta.enumer.EditId_EN;
import gov.cms.hh.data.meta.enumer.EditType_EN;
import gov.cms.hh.data.meta.enumer.PeriodTiming_EN;
import gov.cms.hh.data.meta.enumer.ReferralSource_EN;
import gov.cms.hh.grouper.DataManagerIF;
import gov.cms.hh.logic.specification.IsValidPeriodTiming;
import gov.cms.hh.logic.specification.IsValidReferralSource;
import gov.cms.hh.logic.validation.Edit;
import gov.cms.hh.logic.validation.EditIF;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author CMS
 */
public class HippsPosition1 extends HippsPositionAbstract implements HippsPositionIF {
    
    private static final Logger LOG = Logger.getLogger(HippsPosition1.class.getName());

    /*
    * 1. HIPPS 1st position
    * 1.1 Validate timing
    * 1.2 Validate occurance
    * 1.3 Calculate 1st value
     */

    /**
     *
     * @param claim
     * @param dataManager
     */

    public HippsPosition1(final ClaimIF claim, final DataManagerIF dataManager) {
        super(claim, dataManager);
        setOrder(1);
    }

    /**
     *
     * @return
     */
    @Override
    public boolean validate() {

        final int GRC_INVALID_PERIOD_TIMING = 2;

        boolean retFlag = true;

        // 1.1 Validate period timing
        IsValidPeriodTiming periodTimingSpecs = new IsValidPeriodTiming(getClaim().getPeriodTiming().getValue());
        if (!periodTimingSpecs.isSatisfiedBy(PeriodTiming_EN.values())) {
            getClaim().getPeriodTiming().setValid(false);
            getClaim().getPeriodTiming().addEdit(
                    new Edit(EditId_EN.VALUE_INVALID, getClaim().getPeriodTiming().toString()
                            + EditId_EN.VALUE_INVALID.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
            );
            getClaim().getPeriodTiming().addEdit(
                    new Edit(EditId_EN.VALUE_UNGROUPABLE, getClaim().getPeriodTiming().toString()
                            + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
            );

            // generate GRC "INVALID PERIOD TIMING"
            addReturnCode(GRC_INVALID_PERIOD_TIMING);

            retFlag = false;
        }

        // 1.2 Validate referral occurance
        IsValidReferralSource referralSourceSpecs = new IsValidReferralSource(getClaim().getReferralSource().getValue());
        if (!referralSourceSpecs.isSatisfiedBy(ReferralSource_EN.values())) {
            getClaim().getReferralSource().setValid(false);
            EditIF edit = new Edit(EditId_EN.VALUE_OUT_OF_RANGE,
                    getClaim().getReferralSource().toString() + " is not in range",
                    EditType_EN.INFO, Level.WARNING);
            getClaim().getReferralSource().addEdit(edit);
        }

        return retFlag;
    }

    /**
     *
     * @return
     */
    @Override
    public boolean calculate() {

        // 1.3 Calculate 1st value
        final int HIPPS_POSITION = 1;
        final int HIPPS_VALUE_1 = 1;    // Community Early
        final int HIPPS_VALUE_2 = 2;    // Institutional Early
        final int HIPPS_VALUE_3 = 3;    // Community Late
        final int HIPPS_VALUE_4 = 4;    // Institutional Late

        boolean retFlag = true;
        String text = null;

        String periodTiming = getClaim().getPeriodTiming().getValue();
        String ptDesc = (getClaim().getPeriodTiming().getDescription() == null ? "(no description)" 
                : "(" + getClaim().getPeriodTiming().getDescription()) + ")";
        String referralSource = getClaim().getReferralSource().getValue();
        String rsDesc = (getClaim().getReferralSource().getDescription() == null ? "(no description)" 
                : "(" + getClaim().getReferralSource().getDescription()) + ")";

        // TO DO
        // add description for detail report
        
        boolean isOutOfRange = !getClaim().getReferralSource().isValid();
        int hippsValue = 0;
        
        if (getClaim().getPeriodTiming().isValid()) {

            if (periodTiming.equals("" + PeriodTiming_EN.PERIOD_TIMING_EARLY.getId())) {
                if (isOutOfRange) {
                    hippsValue = HIPPS_VALUE_1;
                } else {
                    hippsValue = HIPPS_VALUE_2;
                }
            }

            if (periodTiming.equals("" + PeriodTiming_EN.PERIOD_TIMING_LATE.getId())) {
                if (isOutOfRange) {
                    hippsValue = HIPPS_VALUE_3;
                } else {
                    hippsValue = HIPPS_VALUE_4;
                }
            }

            // Add logic info edits for Detail Report
            text = "Period Timing = " + periodTiming + " " + ptDesc;
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS1, text);
            
            text = "Referral Source = " 
                    + (referralSource == null ? "(empty)" : referralSource + " " + rsDesc)
                    + (isOutOfRange ? " (Out of range)" : "");
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS1, text);
            
            text = "HIPPS Position 1 = " + hippsValue;
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS1, text);
            
            // Set HIPPS 1
            return setHippsCode(HIPPS_POSITION, "" + hippsValue);

        } else {

            // Add logic info edits for Detail Report
            text = "Period Timing = " 
                    + (periodTiming == null ? "(empty)" : periodTiming  + " " + ptDesc) + " (Incorrect, SEVERE)";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS1, text);

            text = "HIPPS Position 1 = " + hippsValue + " (Not calculated)";
            addHippsLogicMessage(EditId_EN.VALUE_HIPPS1, text);
            
            retFlag = false;
        }

        return retFlag;
    }
    
}
