/* 
 * 
 * Home Health Grouper Software
 * 
 * Center for Medicare and Medicaid Services (CMS)
 * 
 * All code is provided as is.
 * 
 */
package gov.cms.hh.logic.processor;

import gov.cms.hh.claim.ClaimIF;
import gov.cms.hh.claim.OasisResponseIF;
import gov.cms.hh.claim.ReturnCode;
import gov.cms.hh.data.files.TableNames_EN;
import gov.cms.hh.data.meta.enumer.EditId_EN;
import gov.cms.hh.data.meta.enumer.EditType_EN;
import gov.cms.hh.data.meta.enumer.Oasis_EN;
import gov.cms.hh.diagnosis.ClinicalGroup;
import gov.cms.hh.diagnosis.ComorbidityGroup;
import gov.cms.hh.diagnosis.Diagnosis;
import gov.cms.hh.diagnosis.DiagnosisIF;
import gov.cms.hh.diagnosis.DiagnosisSubchapter;
import gov.cms.hh.grouper.DataManagerIF;
import gov.cms.hh.grouper.utility.Utility;
import gov.cms.hh.logic.specification.IsValidOasisResponse;
import gov.cms.hh.logic.validation.Edit;
import gov.cms.hh.logic.validation.EditIF;
import gov.cms.hh.reference.objects.ClinicalGroupObject;
import gov.cms.hh.reference.objects.ComorbidityGroupObject;
import gov.cms.hh.reference.objects.DiagnosisObject;
import gov.cms.hh.reference.objects.DiagnosisSubchapterObject;
import gov.cms.hh.reference.objects.HippsValueObject;
import gov.cms.hh.reference.objects.OasisRespondObject;
import gov.cms.hh.reference.objects.ReturnCodeObject;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author CMS
 */
public abstract class HippsPositionAbstract {

    /**
     *
     * @param dxList
     * @param name
     * @return
     */
    protected DiagnosisIF findDxByComorbidity(final List<DiagnosisIF> dxList, String name) {
        if (dxList == null || dxList.isEmpty() || name == null || name.isEmpty()) {
            return null;
        }
        for (DiagnosisIF dx : dxList) {
            if (!(dx.getEditCollection().hasEditId(EditId_EN.VALUE_INVALID) || // Only valid DXs should be processed
                    dx.getEditCollection().hasEditId(EditId_EN.VALUE_IGNORE))) { // don't count ingnored Dxs
                if (dx.getComorbidityGroup().getName().trim().equalsIgnoreCase(name.trim())) {
                    return dx;
                }
            }
        }
        return null;
    }

    /**
     *
     * @param hippsPosition
     * @param text
     */
    protected void addHippsLogicMessage(EditId_EN hippsPosition, String text) {
        EditIF edit = new Edit(hippsPosition, text, EditType_EN.SPECIFICATION, Level.INFO);
        getClaim().getHippsCode().addEdit(edit);
    }

    /**
     *
     * @return
     */
    protected boolean validateFunctionalFields() {

        boolean valHospRisk = validateRiskHospitalisation();
        if (!valHospRisk) {
            addReturnCode(Oasis_EN.M1033_HOSP_RISK_HSTRY_FALLS.getErrorReturnCode());
        }
        boolean valOther = validateOtherFunctionalFields();
        if (!(valHospRisk & valOther)) {
            claim.getOasisFields().addEdit(
                    new Edit(EditId_EN.VALUE_UNGROUPABLE, "OASIS has invalid or inconsistent value(s). "
                            + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
            );
        }
        return valHospRisk & valOther;
    }

    private String getValidValues(OasisResponseIF oResp, List<OasisRespondObject> refList) {
        List<String> validList = new ArrayList<>();
        for (OasisRespondObject oasRespRef : refList) {
            if (oResp.getType().name().startsWith(oasRespRef.getFieldId().trim().toUpperCase())) {
                validList.add(oasRespRef.getResponse());
            }
        }
        return Utility.join(validList);
    }

    private boolean validateFunctionalField(OasisResponseIF oResp, List<OasisRespondObject> refList) {
        IsValidOasisResponse actualResp = new IsValidOasisResponse(oResp);
        for (OasisRespondObject oasRespRef : refList) {
            if (actualResp.isSatisfiedBy(oasRespRef)) {
                // Keep score here
                // for valid response
                oResp.setScore(Integer.parseInt(oasRespRef.getCategoryPoints()));
                return true;
            }
        }
        return false;
    }

    private boolean validateOtherFunctionalFields() {

        boolean retFlag = true;

        final String PREFIX_NOT = "M1033";

        List<OasisRespondObject> refList = dataManager.getDataMap().get(TableNames_EN.OASIS_Responses).getData();

        List<OasisResponseIF> oResList = claim.getOasisFields().getCollection();

        for (OasisResponseIF oResp : oResList) {
            if (!oResp.getType().name().startsWith(PREFIX_NOT)) {
                // Check against ROT
                if (!validateFunctionalField(oResp, refList)) {
                    oResp.addEdit(
                            new Edit(EditId_EN.VALUE_INVALID, oResp.toString()
                                    + ". Valid values=[" + getValidValues(oResp, refList) + "]", EditType_EN.CONSISTENCY, Level.SEVERE)
                    );
                    // adding return code
                    addReturnCode(oResp.getType().getErrorReturnCode());
                    retFlag = false;
                }
            }
        }

        return retFlag;

    }

    private boolean validateRiskHospitalisation() {

        boolean retFlag = true;

        final String PREFIX = "M1033";
        final List<String> VALID_VALUES = Arrays.asList("0", "1");

        int sum = 0;
        int noneAbove = 0;
        List<OasisResponseIF> oResList = claim.getOasisFields().getCollection();
        for (OasisResponseIF oResp : oResList) {
            if (oResp.getType().name().startsWith(PREFIX)) {
                if (!VALID_VALUES.contains(oResp.getValue())) {
                    oResp.addEdit(
                            new Edit(EditId_EN.VALUE_INVALID, oResp.toString()
                                    + ". Valid values=[" + Utility.join(VALID_VALUES) + "]", EditType_EN.CONSISTENCY, Level.SEVERE)
                    );
                    retFlag &= false;
                } else {
                    int val = Integer.parseInt(oResp.getValue());
                    sum += val;
                    if (oResp.getType().name().equals(Oasis_EN.M1033_HOSP_RISK_NONE_ABOVE.name())) {
                        noneAbove = val;
                    }
                    if (oResp.getType().name().equals(Oasis_EN.M1033_HOSP_RISK_CRNT_EXHSTN.name())
                            || oResp.getType().name().equals(Oasis_EN.M1033_HOSP_RISK_OTHR_RISK.name())) {
                        oResp.addEdit(
                                new Edit(EditId_EN.VALUE_IGNORE, oResp.toString() + " "
                                        + EditId_EN.VALUE_IGNORE.getDescription(), EditType_EN.SPECIFICATION, Level.INFO)
                        );
                    }
                }
            }
        }

        // check "M1033 - None of the above"
        if (noneAbove > 0 && (sum - noneAbove) > 0) {
            claim.getOasisFields().addEdit(
                    new Edit(EditId_EN.VALUE_UNGROUPABLE, "If M1033_HOSP_RISK_NONE_ABOVE = 1, then all items from M1033_HOSP_RISK_HSTRY_FALLS through M1033_HOSP_RISK_OTHR_RISK must = 0. "
                            + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
            );
            retFlag &= false;
        }
        if (sum == 0) {
            claim.getOasisFields().addEdit(
                    new Edit(EditId_EN.VALUE_UNGROUPABLE, "If M1033_HOSP_RISK_NONE_ABOVE = 0, then at least one item from M1033_HOSP_RISK_HSTRY_FALLS through M1033_HOSP_RISK_OTHR_RISK must = 1. "
                            + EditId_EN.VALUE_UNGROUPABLE.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE)
            );
            retFlag &= false;
        }

        return retFlag;

    }

    /**
     * Validate diagnoses
     */
    protected void validateDiagnoses() {
        try {
            // Validate PDX
            DiagnosisIF pdx = claim.getPrimaryDiagnosis();
            validateDiagnosis(pdx);
            // Was PDX valid?
            if (!pdx.isValid()) {
                // make edit severe for invalid pdx
                pdx.addEdit(
                        new Edit(EditId_EN.VALUE_INVALID, pdx.toString()
                                + EditId_EN.VALUE_INVALID.getDescription(), EditType_EN.CONSISTENCY, Level.SEVERE));
            }
            // Validate SDX(s)
            List<DiagnosisIF> sdxList = claim.getSecondaryDiagnoses().getCollection();
            for (int i = 0; i < sdxList.size(); i++) {
                DiagnosisIF diag = sdxList.get(i);
                validateDiagnosis(diag);
            }
        } catch (ParseException ex) {
            Logger.getLogger(HippsPositionAbstract.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    private void validateDiagnosis(DiagnosisIF dx) throws ParseException {
        // Validate DX
        DiagnosisIF dxRef = null;
        // 2.1.1 Replace "shallow" DX object with "full" PDX object if DX value is valid
        dxRef = getDiagnosisReference(dx.getValue());
        if (dxRef == null) {
            dx.setValid(false);
            // Report invalid DX
            dx.addEdit(
                    new Edit(EditId_EN.VALUE_INVALID, dx.toString()
                            + EditId_EN.VALUE_INVALID.getDescription(), EditType_EN.CONSISTENCY, Level.WARNING)
            );
        } else {
            dx.setPrimaryAwarding(dxRef.isPrimaryAwarding());
            dx.setManifestation(dxRef.isManifestation());
            dx.setId(dxRef.getId());
            dx.setECOI(dxRef.isECOI());
            dx.setDescription(dxRef.getDescription());
            dx.setDiagnosisSubchapter(dxRef.getDiagnosisSubchapter());
            dx.setComorbidityGroup(dxRef.getComorbidityGroup());
            dx.setClinicalGroup(dxRef.getClinicalGroup());
            dx.setValid(true);
            dx.addEdit(
                    new Edit(EditId_EN.VALUE_VALID, dxRef.toString()
                            + EditId_EN.VALUE_VALID.getDescription(), EditType_EN.INFO, Level.INFO)
            );
        }

        dx.setVerified(true);
    }

    private DiagnosisIF getDiagnosisReference(String value) throws ParseException {
        DiagnosisIF dx = null;

        // Get DX DTO
        DiagnosisObject dxObj = (DiagnosisObject) dataManager.getDataMap()
                .get(TableNames_EN.Diagnoses).getDataMap().get(value);
        if (dxObj == null) {
            return null; // Invalid DX
        }

        // Get Clinical Group DTO
        ClinicalGroupObject clinicalGroupObj = (ClinicalGroupObject) dataManager.getDataMap()
                .get(TableNames_EN.Clinical_Groups).getDataMap().get(dxObj.getClinicalGroup());
        // Get Clinical Group internal object
        ClinicalGroup clinicalGroup = null;
        if (clinicalGroupObj != null) {
            clinicalGroup = new ClinicalGroup(0, clinicalGroupObj);
        }

        // Get Comorbidity Group DTO
        ComorbidityGroupObject comorbidityGroupObj = (ComorbidityGroupObject) dataManager.getDataMap()
                .get(TableNames_EN.Comorbidity_Groups).getDataMap().get(dxObj.getComorbidityGroup());
        // Get Comorbidity Group internal object
        ComorbidityGroup comorbidityGroup = null;
        if (comorbidityGroupObj != null) {
            comorbidityGroup = new ComorbidityGroup(0, comorbidityGroupObj);
        }

        // Get Diagnosis Subchapter DTO
        List<DiagnosisSubchapterObject> diagSubchapObjList = dataManager.getDataMap()
                .get(TableNames_EN.Diagnosis_Subchapters).getData();

        // Get Diagnosis Subchapter internal object
        DiagnosisSubchapter diagnosisSubchapter = null;

        for (DiagnosisSubchapterObject diagSubchapObj : diagSubchapObjList) {
            String key = diagSubchapObj.getDiagnosisFrom() + "-" + diagSubchapObj.getDiagnosisTo();
            if (key.trim().equalsIgnoreCase(dxObj.getSubChapter().trim())) {
                diagnosisSubchapter = new DiagnosisSubchapter(diagSubchapObj);
                break;
            }
        }

        dx = new Diagnosis(0, dxObj, clinicalGroup, comorbidityGroup, diagnosisSubchapter);

        return dx;
    }

    /**
     *
     * @param position
     * @param value
     * @return
     */
    public boolean setHippsCode(int position, String value) {
        HippsValueObject hipps = getHippsValueReference(position, value);
        if (hipps != null) {
            getClaim().getHippsCode().setValue(hipps);
            return true;
        } else {
            EditIF edit = new Edit(EditId_EN.VALUE_NOT_CALCULATED,
                    "Not able to calculeate HIPPS value in position: " + position, EditType_EN.INFO, Level.SEVERE);
            getClaim().getHippsCode().addEdit(edit);
            return false;
        }
    }

    private HippsValueObject getHippsValueReference(int position, String value) {
        for (Object obj : dataManager.getDataMap().get(TableNames_EN.HIPPS_Structure).getData()) {
            if (((HippsValueObject) obj).getPosition().equals("" + position)
                    && ((HippsValueObject) obj).getValue().equalsIgnoreCase(value)) {
                return (HippsValueObject) obj;
            }
        }
        return null;
    }

    /**
     *
     * @param code
     * @return
     */
    public boolean addReturnCode(int code) {
        // check if RC already set

        ReturnCodeObject grcObj = getReturnCodeReference(code);
        if (grcObj != null) {
            ReturnCode grc = null;
            try {
                grc = new ReturnCode(grcObj);
            } catch (ParseException ex) {
                Logger.getLogger(HippsPositionAbstract.class.getName()).log(Level.SEVERE, null, ex);
            }
            if (!getClaim().getReturnCodes().getCollection().contains(grc)) {
                getClaim().getReturnCodes().getCollection().add(grc);
            }
            return true;
        } else {
            EditIF edit = new Edit(EditId_EN.VALUE_INVALID,
                    "Not able to set Return Code: " + code, EditType_EN.INFO, Level.SEVERE);
            getClaim().getReturnCodes().addEdit(edit);
            return false;
        }
    }

    private ReturnCodeObject getReturnCodeReference(int code) {
        for (Object obj : dataManager.getDataMap().get(TableNames_EN.Return_Codes).getData()) {
            if (Integer.parseInt(((ReturnCodeObject) obj).getCode()) == code) {
                return (ReturnCodeObject) obj;
            }
        }
        return null;
    }

    /**
     *
     * @param claim
     * @param dataManager
     */
    public HippsPositionAbstract(final ClaimIF claim, final DataManagerIF dataManager) {
        this.claim = claim;
        this.dataManager = dataManager;
    }

    /**
     * @return the claim
     */
    public ClaimIF getClaim() {
        return claim;
    }

    /**
     * @return the dataManager
     */
    public DataManagerIF getDataManager() {
        return dataManager;
    }

    private final ClaimIF claim;
    private final DataManagerIF dataManager;
}
