package gov.cms.grouper.snf.component.v100.logic;

import gov.cms.grouper.snf.SnfContext;
import gov.cms.grouper.snf.component.v100.logic.nursing.BscpLogic;
import gov.cms.grouper.snf.component.v100.logic.nursing.ReducedPhysicalFunctionLogic;
import gov.cms.grouper.snf.component.v100.logic.nursing.SpecialCare;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.lego.SnfVersionImpl;
import gov.cms.grouper.snf.model.enums.NursingCmg;
import gov.cms.grouper.snf.model.reader.Rai300;
import gov.cms.grouper.snf.util.ClaimInfo;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <a href="doc-files/mds-3.0-rai-manual-v1.17.1_october_2019.pdf#page=690" class="req">PDPM Payment
 * Component: Nursing</a>
 */
public class NursingLogic extends SnfVersionImpl<NursingCmg> {

  public static final List<Rai300> extensiveServices =
      Arrays.asList(Rai300.O0100E2, Rai300.O0100F2, Rai300.O0100M2);
  public static final Set<Rai300> skinTreatments =
      SnfUtils.toSet(Rai300.M1200C, Rai300.M1200D, Rai300.M1200E, Rai300.M1200G, Rai300.M1200H);
  public static final Set<Rai300> skinTreatments2 = SnfUtils.toSet(Rai300.M1200A, Rai300.M1200B);

  private final ClaimInfo claim;
  private final SpecialCare specialCare;
  private final ReducedPhysicalFunctionLogic physical;
  private final BscpLogic bscp;

  public NursingLogic(ClaimInfo claim, SpecialCare specialCare,
      ReducedPhysicalFunctionLogic physical, BscpLogic bscp) {
    super(claim.getVersion());
    this.claim = claim;
    this.specialCare = specialCare;
    this.physical = physical;
    this.bscp = bscp;
  }


  protected List<Rai300> getCodeServices() {
    List<Rai300> services = NursingLogic.extensiveServices.stream().filter((item) -> {
      boolean result = claim.isCheckedAndNotNull(item);
      return result;
    }).collect(Collectors.toList());

    return services;
  }

  /**
   * Evaluate the resident's
   * <a href="doc-files/mds-3.0-rai-manual-v1.17.1_october_2019.pdf#page=692" class="req">Extensive
   * Services</a> related to tracheostomy care, ventilator or respirator, in isolation or
   * quarantine.
   *
   * @return the applicable NursingCmg, or null if not applicable
   */
  protected NursingCmg evaluateExtensiveServices() {

    List<Rai300> codedServices = this.getCodeServices();
    int nursingFunctionScore = claim.getFunctionScore();

    NursingCmg result = null;
    if (codedServices != null && !codedServices.isEmpty()) {
      if (nursingFunctionScore >= 15) {
        result = this.specialCare.evaluateClinicallyComplex(true);

      } else if (codedServices.contains(Rai300.O0100E2) && codedServices.contains(Rai300.O0100F2)) {
        result = NursingCmg.ES3;
      } else if (codedServices.contains(Rai300.O0100E2) || codedServices.contains(Rai300.O0100F2)) {
        result = NursingCmg.ES2;
      } else if (codedServices.contains(Rai300.O0100M2) && !codedServices.contains(Rai300.O0100E2)
          && !codedServices.contains(Rai300.O0100F2)) {
        result = NursingCmg.ES1;
      } else {
        result = null;
      }
    }

    return SnfContext.trace(result);
  }

  @Override
  public NursingCmg exec() {

    NursingCmg cmg = this.evaluateExtensiveServices();
    if (cmg == null) {
      List<SnfVersionImpl<NursingCmg>> components =
          Arrays.asList(this.specialCare, this.bscp, this.physical);

      for (SnfVersionImpl<NursingCmg> com : components) {
        cmg = com.exec();
        if (cmg != null) {
          break;
        }
      }
    }

    return SnfContext.trace(cmg);
  }

  public ClaimInfo getClaim() {
    return claim;
  }

  public SpecialCare getSpecialCare() {
    return specialCare;
  }

  public ReducedPhysicalFunctionLogic getPhysical() {
    return physical;
  }

  public BscpLogic getBscp() {
    return bscp;
  }

}
